'use client'
import { zodResolver } from "@hookform/resolvers/zod"
import { useForm } from "react-hook-form"
import { z } from "zod"
import {useTranslations} from 'next-intl';

import {
    Dialog,
    DialogContent,
    DialogClose,
    DialogTrigger,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { IoMdClose } from "react-icons/io";

import {
    Form,
    FormControl,
    FormField,
    FormItem,
    FormLabel,
} from "@/components/ui/form"
import { Input } from "@/components/ui/input";
import { toast } from 'sonner';
import { addWhatsapp } from "@/actions/profile/postWaNotif"
import { editWhatsapp } from "@/actions/profile/putWaNotif"
import { useState, useTransition } from "react";

const formSchema = z.object({
    id: z.number(),
    villaId: z.number().nullable(),
    whatsapp: z.string().min(1, 'Whatsapp cannot be empty'),
})

interface Props {
    isEdit: boolean
    villa: any;
    data?: any;
    loading: (value: boolean) => void
    reload: () => void
}

const AddWhatsapp = ({villa, loading, reload, isEdit, data} : Props) => {
    const t = useTranslations('Index');
    const [open, setOpen] = useState(false)
    const [isPending, startTransition] = useTransition()
    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: {
            id: isEdit ? data.id : Number(villa.villa_id),
            villaId: isEdit ? data.id_villa : null,
            whatsapp: isEdit ? data.phone : '',
        },
    })

    const onSubmit = (values: z.infer<typeof formSchema>) => {
        setOpen(false)
        loading(true)
        startTransition(async() => {
            try {
                let res = null;
                if(isEdit){
                    res = await editWhatsapp(values.id, values.villaId, values.whatsapp);
                }else{
                    res = await addWhatsapp(values.id, values.whatsapp);
                }
                if(res.success){
                    form.reset()
                    loading(false)
                    reload()
                    toast.success(res.message)
                }else{
                    toast.error(res.message)
                }
            } catch (error) {
                loading(false)
                toast.error('Smething went wrong !! Cant add the whatsapp !!')
            }
        })
    }
    return (
        <Dialog open={open} onOpenChange={setOpen} defaultOpen={false}>
            <DialogTrigger asChild>
                {isEdit ? (
                    <Button variant='ghost' className="w-full">
                        {t('edit')}
                    </Button>
                ) : (
                    <Button id="profile-8" size='sm'>
                        {t('addWa')}
                    </Button>
                )}
            </DialogTrigger>
            <DialogContent>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-3">
                        <FormField
                            control={form.control}
                            name="id"
                            render={({ field }) => (
                                <FormItem>
                                    <FormControl>
                                        <Input 
                                            {...field}
                                            className="hidden"/>
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="whatsapp"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>Whatsapp</FormLabel>
                                    <FormControl>
                                        <Input 
                                            {...field}  
                                            placeholder="Ex : +628123456789"
                                            />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <DialogClose asChild>
                            <Button type="submit">
                                {isEdit ? t('editWa') : t('addWa')}
                            </Button>
                        </DialogClose>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    )
}

export default AddWhatsapp