'use client'

import { zodResolver } from "@hookform/resolvers/zod"
import { useForm } from "react-hook-form"
import { z } from "zod"

import { useEffect, useState } from 'react'
import { IoIosSend } from 'react-icons/io'
import {
    Sheet,
    SheetContent,
    SheetDescription,
    SheetHeader,
    SheetTitle,
    SheetTrigger,
} from "@/components/ui/sheet"

import { Button } from "@/components/ui/button"
import {
    Form,
    FormControl,
    FormDescription,
    FormField,
    FormItem,
    FormLabel,
    FormMessage,
} from "@/components/ui/form"
import {
    Popover,
    PopoverContent,
    PopoverTrigger,
} from "@/components/ui/popover"
import { Checkbox } from "@/components/ui/checkbox"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { RxQuestionMarkCircled } from "react-icons/rx"
import Link from "next/link"


const ACCEPTED_IMAGE_TYPES = ["image/jpeg", "image/jpg", "image/png", "image/webp"];

const formSchema = z.object({
    feed: z.string().min(1, 'Please give us your feedback'),
    image: z
        .instanceof(File)
        .nullable()
        .refine(
            (file) => file == null || ACCEPTED_IMAGE_TYPES.includes(file.type),
            "Only .jpg, .jpeg, .png and .webp formats are supported."
        ),
})

const Feedback = () => {

    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: {
            feed: "",
            image: null,
        },
    })

    const onSubmit = (values: z.infer<typeof formSchema>) => {
        console.log(values)
    }

    return (
        <Sheet>
            <SheetTrigger asChild>
                <div className="flex items-center gap-2 cursor-pointer py-2 px-3 hover:bg-stone-200">
                    <IoIosSend className="h-5 w-5"/>
                    <span className="text-sm">Send Feedback</span>
                </div>
            </SheetTrigger>
            <SheetContent>
                <SheetHeader>
                    <SheetTitle>Send your feedback</SheetTitle>
                    <SheetDescription className="sr-only">
                    </SheetDescription>
                </SheetHeader>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-8 mt-6">
                        <FormField
                            control={form.control}
                            name="feed"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel className="font-normal">Describe your feedback</FormLabel>
                                    <FormControl>
                                        <Textarea placeholder="Tell us what prompted this feedback..." className="resize-y" {...field} />
                                    </FormControl>
                                    <div className="flex items-start gap-1">
                                        <FormDescription>
                                            Please don’t include any sensitive information
                                        </FormDescription>
                                        <Popover>
                                            <PopoverTrigger>
                                                <RxQuestionMarkCircled className="text-muted-foreground"/>
                                            </PopoverTrigger>
                                        <PopoverContent className="text-sm">Sensitive information is any data that should be protected. For example, don’t include passwords, credit card numbers, and personal details.</PopoverContent>
                                        </Popover>
                                    </div>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="image"
                            render={({ field: { value, onChange, ...fieldProps } }) => (
                                <FormItem>
                                    <FormLabel className="font-normal">A screenshot will help us better understand your feedback. (optional)</FormLabel>
                                    <FormControl>
                                        <Input 
                                            {...fieldProps} 
                                            type="file"
                                            accept="image/*, application/pdf"
                                            onChange={(event) => {
                                                const displayUrl = URL.createObjectURL(event.target.files![0])
                                                const selectedFile = event.target.files![0];
                                                if (selectedFile) {
                                                    form.setValue('image', selectedFile);
                                                } else {
                                                    form.setValue('image', null);
                                                }
                                            }}/>
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <div className="space-y-3 mt-auto">
                            <div className="flex items-center space-x-2">
                                <Checkbox id="terms" />
                                <label
                                    htmlFor="terms"
                                    className="text-sm font-medium leading-none text-muted-foreground peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                                >
                                    We may email you for more information or updates
                                </label>
                            </div>
                            <p className="text-sm text-muted-foreground">
                                Some{" "}
                                <Link
                                    href="#"
                                    className="underline underline-offset-4 hover:text-primary"
                                >
                                account and system information
                                </Link>{" "}
                                may be sent to Google. We will use it to fix problems and improve our services, subject to our{" "}
                                <Link
                                    href="#"
                                    className="underline underline-offset-4 hover:text-primary"
                                >
                                Privacy Policy
                                </Link>{" "}
                                and{" "}
                                <Link
                                    href="#"
                                    className="underline underline-offset-4 hover:text-primary"
                                >
                                Terms of Service
                                </Link>{" "}
                                We may email you for more information or updates. Go to{" "}
                                <Link
                                    href="#"
                                    className="underline underline-offset-4 hover:text-primary"
                                >
                                Legal Help
                                </Link>{" "}
                                to ask for content changes for legal reasons.
                            </p>
                        </div>
                        <div className="flex items-center justify-end">
                            <Button type="submit">Send</Button>
                        </div>
                    </form>
                </Form>
            </SheetContent>
        </Sheet>


    )
}

export default Feedback