import NextAuth from "next-auth"

import Credentials from "next-auth/providers/credentials"

export const {
    handlers: { GET, POST},
    auth,
    signIn,
    signOut,
} = NextAuth({
    trustHost: process.env.AUTH_TRUST_HOST === "true",
    session: { strategy: "jwt"},
    pages: {
        signIn: "/login",
    },
    providers: [
        Credentials({
            credentials: {
                email: { label: "Email", type: "email" },
                password: {  label: "Password", type: "password" }
            },
            async authorize(credentials){
                const csrfUrl = process.env.CSRF_COOKIE_URL
                const res = await fetch(`${csrfUrl}`, {
                    method: "GET",
                })

                const setCookieHeader = res.headers.get("set-cookie")
                // console.log("setCookieHeader", setCookieHeader)
                // you'll find your_site_session key in this console log

                const cookies = setCookieHeader?.split(", ")
                // console.log(cookies)
                let sessionKey = null
                let xsrfToken = null

                for (const cookie of cookies!) {
                if (cookie.startsWith("laravel_session=")) {
                    sessionKey = cookie.split("=")[1]
                } else if (cookie.startsWith("XSRF-TOKEN=")) {
                    xsrfToken = cookie.split("=")[1]
                }

                if (sessionKey && xsrfToken) {
                    break
                }
                }
                const data = {
                    email: credentials?.email,
                    password: credentials?.password,
                }
                const headers = new Headers({
                    Cookie: `laravel_session=${sessionKey}`,
                    "Content-Type": "application/json",
                    "Accept": "application/json",
                })

                if (xsrfToken) {
                    headers.append("X-XSRF-TOKEN", xsrfToken)
                }

                const options = {
                    method: "POST",
                    headers,
                    body: JSON.stringify(data),
                }
                try{
                    const loginUrl = process.env.LOGIN_URL
                    const response = await fetch(`${loginUrl}`, options);
    
                    if (response.ok) {
                        const data = await response.json()
                        return {
                            user: data.user,
                            accessToken: data.access_token,
                        };
                    } else {
                        console.log("HTTP error! Status:", response.status)
                        // Handle non-successful response here, return an appropriate JSON response.
                        return null
                    }
                } catch (error) {
                    console.log("Error", error)
                }

                return null
            }
        })
    ],
    callbacks: {
        async jwt({ token, account, user }) {
            if (user) {
                token.user = user;
                token.accessToken = user.accessToken;
            }
            return token
        },
        async session({ session, token }) {
            session.user = token.user;
            session.accessToken = token.accessToken as string;
            return session;
        },
    },
})
