'use client'

import { z } from "zod"
import { zodResolver } from "@hookform/resolvers/zod"
import { useForm } from "react-hook-form"
import { cn } from "@/lib/utils"
import { useState, useTransition } from "react"
import { login } from "@/actions/auth/login"
import { useRouter } from "next/navigation"

import {
    Form,
    FormControl,
    FormDescription,
    FormField,
    FormItem,
    FormLabel,
    FormMessage,
} from "@/components/ui/form"

import { Label } from "@/components/ui/label"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import { Checkbox } from "@/components/ui/checkbox"
import FormError from "@/components/web/form-error"

import { LuLoader2 } from "react-icons/lu";
import { FaEyeSlash } from "react-icons/fa6"
import { FaEye } from "react-icons/fa";
import { AiOutlineLoading3Quarters } from "react-icons/ai";

import { loginSchema } from "@/schemas"
import Link from "next/link"

const UserAuthForm = () => {
    const router = useRouter()
    const [loading, setLoading] = useState<boolean>(false);
    const [pass, setPass] = useState<boolean>(false);
    const [dataError, setDataError] = useState<string | undefined>("")
    const [isPending, startTransition] = useTransition();
    const form = useForm<z.infer<typeof loginSchema>>({
        resolver: zodResolver(loginSchema),
        defaultValues: {
            email: '',
            password: ''
        },
    })

    const onSubmit = async(values: z.infer<typeof loginSchema>) => {
        setDataError("")
        setLoading(true)
        startTransition(() => {
            login(values)
                .then((data) => { 
                    if(data.success){ 
                        router.push('/')
                    }else{
                        setDataError(data.error)
                        setLoading(false)
                    }
                })
        })
    }

    const showPassword = () => {
        setPass(!pass)
    }

    return (
        <div className="grid gap-6">
            <Form {...form}>
                <form onSubmit={form.handleSubmit(onSubmit)}>
                    <FormError data={dataError}/>
                    <div className="grid gap-3">
                        <div className="grid gap-1">
                            <FormField
                                control={form.control}
                                name="email"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Email</FormLabel>
                                        <FormControl>
                                            <Input
                                                {...field}
                                                id="email"
                                                placeholder="name@example.com"
                                                type="email"
                                                autoCapitalize="none"
                                                autoComplete="email"
                                                autoCorrect="off"
                                                disabled={loading}
                                            />
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                        </div>
                        <div className="grid gap-1">
                        <FormField
                            control={form.control}
                            name="password"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>Password</FormLabel>
                                    <FormControl>
                                        <div className="relative h-full">
                                            <Input placeholder="Your password" type={`${pass ? 'text' : 'password'}`} {...field} className="w-80 max-w-80 " disabled={loading}/>
                                            <div className="absolute top-0 right-0 h-full">
                                                <Button variant="outline" type="button" onClick={showPassword} disabled={loading}>
                                                    {pass ? <FaEyeSlash /> : <FaEye />}
                                                </Button>
                                            </div>
                                        </div>
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        </div>
                        <p className="px-8 text-center text-sm text-muted-foreground">
                            By clicking continue, you agree to our{" "}
                            <Link
                            href="#"
                            className="underline underline-offset-4 hover:text-primary"
                            >
                            Terms of Service
                            </Link>{" "}
                            and{" "}
                            <Link
                            href="#"
                            className="underline underline-offset-4 hover:text-primary"
                            >
                            Privacy Policy
                            </Link>
                            .
                        </p>
                        <Button type="submit" disabled={loading}>
                            {loading && (
                                <div className="animate-spin mr-2 align-middle">
                                    <AiOutlineLoading3Quarters />
                                </div>
                            )}
                            Login
                        </Button>
                    </div>
                </form>
            </Form>
        </div>
    )
}

export default UserAuthForm