"use client"


import {
    ColumnDef,
    flexRender,
    ColumnFiltersState,
    SortingState,
    getCoreRowModel,
    useReactTable,
    getSortedRowModel,
    getPaginationRowModel,
    getFilteredRowModel,
} from "@tanstack/react-table"
import {useTranslations} from 'next-intl';
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow,
} from "@/components/ui/table"
import { Button } from "@/components/ui/button"
import { useState } from "react"
import {
    Card,
    CardContent,
    CardDescription,
    CardHeader,
    CardTitle,
} from "@/components/ui/card"
import BookingsPopover from "@/components/popover/bookings-popover"
import { Skeleton } from "@/components/ui/skeleton"
import { LuChevronsLeft, LuChevronsRight } from "react-icons/lu";


interface DataTableProps<TData, TValue> {
    columns: ColumnDef<TData, TValue>[]
    data: TData[]
    onClick: ( value: any) => void
    loading: boolean
}

export function DataTable<TData, TValue>({
    columns,
    data,
    onClick,
    loading
}: DataTableProps<TData, TValue>) {
    const t = useTranslations('Index');
    const [sorting, setSorting] = useState<SortingState>([])
    const [columnFilters, setColumnFilters] = useState<ColumnFiltersState>([])
    const table = useReactTable({
        data,
        columns,
        getCoreRowModel: getCoreRowModel(),
        getPaginationRowModel: getPaginationRowModel(),
        getFilteredRowModel: getFilteredRowModel(),
        onColumnFiltersChange: setColumnFilters,
        onSortingChange: setSorting,
        getSortedRowModel: getSortedRowModel(),
        state: {
            sorting,
            columnFilters,
        },
    })

    return (
        <Card className="drop-shadow-lg">
            <CardHeader>
                <div id="booking-1" className="flex items-center w-fit">
                    <CardTitle>{t('bookingList')}</CardTitle>
                    <div className="ml-2">
                        <BookingsPopover />
                    </div>
                </div>
                <CardDescription>{t('pb1')} {table.getFilteredRowModel().rows.length} {t('pb2')}</CardDescription>
            </CardHeader>
            <CardContent className="overflow-x-auto">
                <Table>
                    <TableHeader>
                        {table.getHeaderGroups().map((headerGroup) => (
                            <TableRow key={headerGroup.id}>
                            {headerGroup.headers.map((header) => {
                                return (
                                <TableHead key={header.id}>
                                    {header.isPlaceholder
                                    ? null
                                    : flexRender(
                                        header.column.columnDef.header,
                                        header.getContext()
                                        )}
                                </TableHead>
                                )
                            })}
                            </TableRow>
                        ))}
                    </TableHeader>
                    <TableBody id="booking-9">
                        {loading ? (
                            <TableRow>
                                <TableCell>
                                    <Skeleton className="h-5 min-w-[180px] rounded-full" />
                                </TableCell>
                                <TableCell>
                                    <div className="space-y-1">
                                        <Skeleton className="h-5 min-w-[180px] rounded-full" />
                                        <Skeleton className="h-5 min-w-[180px] rounded-full" />
                                    </div>
                                </TableCell>
                                <TableCell>
                                    <Skeleton className="h-5 min-w-[117px] rounded-full" />
                                </TableCell>
                                <TableCell>
                                    <Skeleton className="h-5 min-w-[115px] rounded-full" />
                                </TableCell>
                                <TableCell>
                                    <Skeleton className="h-14 w-14 rounded-lg" />
                                </TableCell>
                                <TableCell>
                                    <Skeleton className="h-5 min-w-[158px] rounded-full" />
                                </TableCell>
                                <TableCell>
                                    <Skeleton className="h-5 min-w-[158px] rounded-full" />
                                </TableCell>
                            </TableRow>
                        ) : (
                            table.getRowModel().rows?.length ? (
                                table.getRowModel().rows.map((row) => (
                                <TableRow
                                    key={row.id}
                                    data-state={row.getIsSelected() && "selected"}
                                    onClick={() => onClick(row.original)}
                                    className="cursor-pointer"
                                >
                                    {row.getVisibleCells().map((cell) => (
                                    <TableCell key={cell.id}>
                                        {flexRender(cell.column.columnDef.cell, cell.getContext())}
                                    </TableCell>
                                    ))}
                                </TableRow>
                                ))
                            ) : (
                                <TableRow>
                                    <TableCell colSpan={columns.length} className="h-24 text-center">
                                        No results.
                                    </TableCell>
                                </TableRow>
                            )
                        )}
                    </TableBody>
                </Table>
                <div className="flex items-center justify-end space-x-2 py-4">
                    
                    <div id="booking-10" className="space-x-2">
                        <Button
                            variant="outline"
                            className="sm h-9"
                            onClick={() => table.firstPage()}
                            disabled={!table.getCanPreviousPage()}
                        >
                            First
                        </Button>
                        <Button
                            variant="outline"
                            size="sm"
                            onClick={() => table.previousPage()}
                            disabled={!table.getCanPreviousPage()}
                        >
                            Prev
                        </Button>
                        <Button
                            variant="outline"
                            size="sm"
                            onClick={() => table.nextPage()}
                            disabled={!table.getCanNextPage()}
                        >
                            Next
                        </Button>
                        <Button
                            variant="outline"
                            className="sm h-9"
                            onClick={() => table.lastPage()}
                            disabled={!table.getCanNextPage()}
                        >
                            Last
                        </Button>
                    </div>
                </div>
            </CardContent>
        </Card>
    )
}
