'use client'

import { Suspense, useEffect, useState } from 'react';
import {useTranslations} from 'next-intl';
import { format, getMonth, getYear } from 'date-fns';
import Link from 'next/link';

import { LuLayoutDashboard, LuGauge } from "react-icons/lu";
import { GoCalendar } from "react-icons/go";
import { IoBedOutline, IoPersonCircleOutline, IoBarChartSharp } from "react-icons/io5";

import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from '@/components/ui/button';

import VillaSwitcher from '@/components/dashboard/villa-switcher';
import { UserNav } from '@/components/dashboard/user-nav';
import Loading from '@/components/web/loading';
import PeriodPicker from '@/components/dashboard/overview/period-picker';
import { DataRevenue } from '@/components/dashboard/overview/data-revenue';
import ChartWrapper from '@/components/dashboard/overview/chart-wrapper';
import BookingWrapper from '@/components/dashboard/overview/booking-wrapper';
import CockpitWrapper from '@/components/dashboard/cockpit/cockpit-wrapper';

import MonthReportPopover from "@/components/popover/month-report";
import PeriodPopover from "@/components/popover/period-changer";
import { changeDash } from '@/actions/data/changeDash';
import { toast } from 'sonner';
import Tutorial from './tutorial/main';
import AllBooking from './bookings/all-booking';
import Report from './report/report-main';
import Profile from './profile-menu/profile';
import Test from '../debug/test';
import Footer from '../web/footer';

const date = new Date(); // Current date
const monthIndex = getMonth(date); // Month index (0-11)
const monthNumber = monthIndex + 1; // Month number (1-12)
const yearNumber = getYear(date); // Year number
const lastMonthDate = new Date(date.getFullYear(), date.getMonth() - 1, 1);


interface Props {
    users: any;
    villas: any;
    tokens: any;
    defaultDash: any;
    defaultBooking: any;
    defaultCalendar: any;
    defaultLive: any;
    defaultMetrics: any;
}
const MainDashboard = ({users, villas, tokens, defaultDash, defaultBooking, defaultCalendar, defaultLive, defaultMetrics}: Props) => {
    const [villa, setVilla] = useState<any>(villas)
    const [selectedVilla, setSelectedVilla] = useState<any>(villas[0])
    const [selectedBooking, setSelectedBooking] = useState<any>(null)
    const [isLoading, setIsLoading] = useState(false)
    const [dash, setDash] = useState<any>(defaultDash)
    const [booking, setBooking] = useState<any>(defaultBooking)
    const [month, setMonth ] = useState<number>(monthNumber)
    const [year, setYear] = useState<number>(yearNumber)
    const [startTour, setStartTour] = useState(false)
    const [active, setActive] = useState('overview')
    const t = useTranslations('Index');

    const handleChangeVilla = (v: any) => {
        setSelectedVilla(v)
    }

    const handleIsChange  = async (value: boolean, m: number, y: number) => {
        if (value === true) {
            setIsLoading(true)
            const res = await changeDash(selectedVilla.room_id, m, y )
            console.log(res)
            if(res.success){
                setDash(res)
                setMonth(res.data.currentReports)
                setYear(res.data.currentReportYear)
                setIsLoading(false)
            }else{
                setTimeout(() => {
                    setDash(defaultDash)
                    setMonth(m)
                    setYear(y)
                    setIsLoading(false)
                    toast.error(`${t('noReportFor')} ${res.month} ${y}`)
                }, 3000)
            }
        }
    }

    useEffect(() => {
        const isTour = localStorage.getItem('tour')
        if(!isTour || isTour === 'false'){
            setStartTour(true)
        }
    },[])

    const handleSelectedBooking = (value: any) => {
        setSelectedBooking(value)
    }

    const handleStartTour = () => {
        setStartTour(true)
    }

    const handleEndTour = () => {
        setStartTour(false)
        localStorage.setItem('tour', 'true')
    }
    return (
        <>
            <main className="min-h-screen lg:container w-full mx-auto">
                <div className="flex flex-col h-full">
                    {startTour && <Tutorial start={startTour} setStartTour={setStartTour} onTourEnd={handleEndTour} active={active}/>}
                    <div className="flex h-16 items-center px-4 gap-4 border-b">
                        <Suspense fallback={<Loading />}>
                            <VillaSwitcher villa={villa} loading={isLoading} onchange={handleChangeVilla} selected={selectedVilla}/>
                        </Suspense>
                        <div className="ml-auto flex items-center space-x-4">
                            <Suspense fallback={<Loading />}>
                                <UserNav data={users} loading={isLoading} setLoading={setIsLoading}/>
                            </Suspense>
                        </div>
                    </div>
                    {/* <div className="flex-1 space-y-4 lg:p-8 p-4 pt-6">
                        <div className="flex lg:flex-row flex-col lg:items-center items-start lg:justify-between space-y-2">
                            <h2 className="text-3xl font-bold tracking-tight">{t('dashboard')}</h2>
                        </div>
                    </div> */}
                    <Tabs onValueChange={setActive} defaultValue="overview" className="space-y-4 my-4 px-4">
                        <TabsList className="bg-transparent flex justify-start">
                            <div id="step-3" className="flex flex-nowrap overflow-x-auto space-x-2 py-2">
                                <TabsTrigger value="overview" disabled={isLoading} className="flex items-center gap-x-1 border drop-shadow-lg transition-transform duration-300 transform hover:-translate-y-2">
                                    <LuLayoutDashboard />
                                    {t('overview')}
                                </TabsTrigger>
                                <TabsTrigger value="cokcpit" disabled={isLoading} className="flex items-center gap-x-1 border drop-shadow-lg transition-transform duration-300 transform hover:-translate-y-2">
                                    <LuGauge />
                                    Live Data
                                </TabsTrigger>
                                {/* <TabsTrigger value="calendar" className="flex items-center gap-x-1 border drop-shadow-lg transition-transform duration-300 transform hover:-translate-y-2">
                                    <GoCalendar />
                                    {t('calendar')}
                                </TabsTrigger> */}
                                <TabsTrigger value="bookings" disabled={isLoading} className="flex items-center gap-x-1 border drop-shadow-lg transition-transform duration-300 transform hover:-translate-y-2">
                                    <IoBedOutline />
                                    {t('bookings')}
                                </TabsTrigger>
                                <TabsTrigger value="report" disabled={isLoading} className="flex items-center gap-x-1 border drop-shadow-lg transition-transform duration-300 transform hover:-translate-y-2">
                                    <IoBarChartSharp />
                                    {t('report')}
                                </TabsTrigger>
                                <TabsTrigger value="profile" disabled={isLoading} className="flex items-center gap-x-1 border drop-shadow-lg transition-transform duration-300 transform hover:-translate-y-2">
                                    <IoPersonCircleOutline />
                                    {t('profile')}
                                </TabsTrigger>
                            </div>
                        </TabsList>
                        <TabsContent value="overview" className="space-y-4 relative">
                            <div className="flex lg:flex-row md:flex-row flex-col lg:items-center items-start justify-between gap-3 lg:py-0 py-3">
                                <div id='step-4' className="flex items-center">
                                    <Suspense fallback={<Loading />}>
                                        <h3 className="text-xl font-semibold leading-none tracking-tight">{t('reports')} ( {dash && dash.data.currentlasreport} )</h3>
                                    </Suspense>
                                    <div className="ml-2">
                                        <MonthReportPopover />
                                    </div>
                                </div>
                                <div className="flex items-center justify-center space-x-2">
                                    <PeriodPopover />
                                    <Suspense fallback={<Loading />}>
                                        <PeriodPicker month={dash ? dash.data.currentReports : month} year={dash ? dash.data.currentReportYear : year} setYear={setYear} setMonth={setMonth} onChange={handleIsChange} loading={isLoading}/>
                                        <Button id='step-6' asChild >
                                            {dash.data.report === '#' ? (
                                                <span>{t('download')}</span>
                                            ) : (
                                                <Link href={dash ? dash.data.report : '#'} target="_blank" download={`${selectedVilla.villa_name}-${dash && dash.data.currentlasreport}-${year}.pdf`}>
                                                    {t('download')}
                                                </Link>
                                            )}
                                        </Button>
                                    </Suspense>
                                </div>
                            </div>
                            <DataRevenue dash={dash.data} lastmonth={dash ? dash.data.lastMonth : format(lastMonthDate, 'MMMM')} loading={isLoading}/>
                            <div className="grid gap-4 lg:grid-cols-7">
                                <ChartWrapper dash={dash.data} year={yearNumber} isLoading={isLoading} chart={dash.revenue}/>
                                <BookingWrapper booking={booking} isLoading={isLoading} selected={selectedVilla}/>
                            </div>
                        </TabsContent>
                        <TabsContent value="cokcpit" className="space-y-4 relative">
                            <CockpitWrapper event={defaultCalendar.events} villa={defaultCalendar.resource}  live={defaultLive} metrics={defaultMetrics} isLoading={isLoading}/>
                        </TabsContent>
                        <TabsContent value="bookings" className="space-y-4 relative">
                            <AllBooking data={defaultCalendar.events} selected={selectedVilla}/>
                        </TabsContent>
                        <TabsContent value="report" className="space-y-4 relative">
                            <Report dash={dash} selected={selectedVilla} year={dash ? dash.currentReportYear : yearNumber} loading={isLoading}/>
                        </TabsContent>
                        <TabsContent value="profile" className="space-y-4 relative">
                            <Profile villa={villa} user={users} token={tokens} selected={selectedVilla} onChange={setIsLoading} reload={() => {}} loading={isLoading}/>
                        </TabsContent>
                    </Tabs>
                </div>
            </main>
            <Footer startTour={handleStartTour} />
        </>
    )
}

export default MainDashboard