'use client'
import { zodResolver } from "@hookform/resolvers/zod"
import { useForm } from "react-hook-form"
import { z } from "zod"
import {useTranslations} from 'next-intl';

import {
    Dialog,
    DialogContent,
    DialogClose,
    DialogTrigger,
    DialogHeader,
    DialogTitle,
    DialogDescription,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { IoMdClose } from "react-icons/io";

import {
    Form,
    FormControl,
    FormField,
    FormItem,
    FormLabel,
} from "@/components/ui/form"
import { Input } from "@/components/ui/input";
import { toast } from 'sonner';
import { useState, useTransition } from "react"
import { addEmail } from "@/actions/profile/postEmailNotif"
import { editEmail } from "@/actions/profile/putEmailNotif";

const formSchema = z.object({
    id: z.number(),
    villaId: z.number().nullable(),
    email: z.string().email({ message: "Please use a valid email" }),
})

interface Props {
    isEdit: boolean
    data?: any;
    villa: any;
    loading: (value: boolean) => void
    reload: () => void
}

const AddEmail = ({villa, loading, reload, isEdit, data } : Props) => {
    const t = useTranslations('Index');
    const [open, setOpen] = useState(false)
    const [isPending, startTransition] = useTransition()
    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: {
            id: isEdit ? data.id : Number(villa.villa_id),
            villaId: isEdit ? data.id_villa : null,
            email: isEdit ? data.mail : '',
        },
    })

    const onSubmit = (values: z.infer<typeof formSchema>) => {
        setOpen(false)
        loading(true)
        startTransition(async() => {
            try {
                let res = null;
                if(isEdit){
                    res = await editEmail(values.id, values.villaId, values.email);
                }else{
                    res = await addEmail(values.id, values.email);
                }
                if(res.success){
                    form.reset()
                    reload()
                    toast.success(res.message)
                }else{
                    toast.error(res.message)
                }
            } catch (error) {
                loading(false)
                toast.error('Smething went wrong, Cant add the email')
            }
        })
    }
    return (
        <Dialog open={open} onOpenChange={setOpen} defaultOpen={false}>
            <DialogTrigger asChild>
                {isEdit ? (
                    <Button variant='ghost' className="w-full">
                        {t('edit')}
                    </Button>
                ) : (
                    <Button id="profile-6" size='sm'>
                        {t('addEmail')}
                    </Button>
                )}
            </DialogTrigger>
            <DialogContent>
                <DialogHeader className="sr-only">
                    <DialogTitle></DialogTitle>
                    <DialogDescription></DialogDescription>
                </DialogHeader>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-3">
                        <FormField
                            control={form.control}
                            name="id"
                            render={({ field }) => (
                                <FormItem>
                                    <FormControl>
                                        <Input 
                                            {...field}
                                            className="hidden"/>
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="email"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>{t('email')}</FormLabel>
                                    <FormControl>
                                        <Input 
                                            {...field}  
                                            placeholder="Ex : example@mail.com"
                                            />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <Button type="submit">
                            {isEdit ? t('editEmail') : t('addEmail')}
                        </Button>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>

    )
}

export default AddEmail