'use client'
import { zodResolver } from "@hookform/resolvers/zod"
import { useForm } from "react-hook-form"
import { z } from "zod"
import {useTranslations} from 'next-intl';

import {
    Dialog,
    DialogClose,
    DialogContent,
    DialogDescription,
    DialogHeader,
    DialogTitle,
    DialogTrigger,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { IoMdClose } from "react-icons/io";

import {
    Form,
    FormControl,
    FormDescription,
    FormField,
    FormItem,
    FormLabel,
    FormMessage,
} from "@/components/ui/form"
import { Input } from "@/components/ui/input";
import { toast } from 'sonner';
import { deleteEmail } from "@/actions/profile/deleteEmailNotif"
import { useState, useTransition } from "react";

const formSchema = z.object({
    id: z.string().min(1),
})

interface Props {
    mail: any;
    loading: (value: boolean) => void
    reload: (value: boolean) => void
}

const DeleteEmail = ({mail, loading, reload} : Props) => {
    const t = useTranslations('Index');
    const [open, setOpen] = useState(false)
    const [isPending, startTransition] = useTransition()
    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: {
            id: `${mail.id}`
        },
    })

    const onSubmit = (values: z.infer<typeof formSchema>) => {
        setOpen(false)
        loading(true)
        startTransition(async() => {
            try {
                const del = await deleteEmail(values.id);
                if(del.success){
                    loading(false)
                    reload(true)
                }
            } catch (error) {
                loading(false)
                toast.error('Smething went wrong , Cant delete the Email')
            }
        })
    }
    return (
        <Dialog open={open} onOpenChange={setOpen} defaultOpen={false}>
            <DialogTrigger asChild>
                <Button variant="ghost" className="w-full">
                    {t('delete')}
                </Button>
            </DialogTrigger>
            <DialogContent>
                <DialogHeader>
                <DialogTitle>{t('qm1')}</DialogTitle>
                <DialogDescription>
                    {t('qm2')} <strong>{mail.mail}</strong> {t('qm3')}
                </DialogDescription>
                </DialogHeader>
                <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-3">
                        <FormField
                            control={form.control}
                            name="id"
                            render={({ field }) => (
                                <FormItem>
                                    <FormControl>
                                        <Input 
                                            {...field}
                                            className="hidden"/>
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        <div className="flex justify-end items-center gap-6">
                            <DialogClose asChild>
                                <Button type="submit">
                                    {t('deleteEmail')}
                                </Button>
                            </DialogClose>
                            <DialogClose asChild>
                                <Button type="button" variant="destructive">
                                    {t('cancel')}
                                </Button>
                            </DialogClose>
                        </div>
                    </form>
                </Form>
            </DialogContent>
        </Dialog>
    )
}

export default DeleteEmail