'use client'
import { useEffect, useState, useTransition } from "react";
import { zodResolver } from "@hookform/resolvers/zod"
import { useForm } from "react-hook-form"
import { z } from "zod"

import ProfilePopover from "@/components/popover/profile-popover"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Skeleton } from "@/components/ui/skeleton"
import {useTranslations} from 'next-intl';
import ListVilla from "./list-villa";
import {
    Form,
    FormControl,
    FormDescription,
    FormField,
    FormItem,
    FormLabel,
    FormMessage,
} from "@/components/ui/form"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { CgMenuRight } from "react-icons/cg";
import { RiEditFill } from "react-icons/ri";
import { getCurrentUser } from "@/actions/profile/getUser";
import AddEmail from "./add-email";
import Loader from "@/components/web/loader";
import DeleteEmail from "./delete-email";
import AddWhatsapp from "./add-whatsapp";
import DeleteWhatsapp from "./delete-whatsapp";
import { PopoverClose } from "@radix-ui/react-popover";
import { updateUser } from "@/actions/profile/putUser";
import { logout } from "@/actions/auth/logout";
import { toast } from 'sonner';
import { useRouter } from "next/navigation";
import { IoCamera } from "react-icons/io5";

const ACCEPTED_IMAGE_TYPES = ["image/jpeg", "image/jpg", "image/png", "image/webp"];

const formSchema = z.object({
    id: z.number(),
    email: z.string().email({ message: "Please use a valid email" }),
    phone: z.string().min(1, 'Phone cannot be empty'),
    password: z.string().optional(),
    image: z
        .instanceof(File)
        .nullable()
        .refine(
            (file) => file == null || ACCEPTED_IMAGE_TYPES.includes(file.type),
            "Only .jpg, .jpeg, .png and .webp formats are supported."
        ),
})

interface Props {
    villa: any;
    user: any;
    selected: any;
    onChange: (value: boolean) => void;
    reload: (value: boolean) => void;
    loading: boolean;
    token: any
}

const Profile = ({villa, user, selected, onChange, reload, loading, token} : Props) => {
    const router = useRouter()
    const [dataUser , setDataUser] = useState<any>(user)
    const [villaImage, setVillaImage] = useState(user.profile);
    const [edit, setEdit] = useState(false);
    const [isLoading, setIsLoading] = useState(loading)
    const t = useTranslations('Index');
    const [isPending, startTransition] = useTransition()

    const form = useForm<z.infer<typeof formSchema>>({
        resolver: zodResolver(formSchema),
        defaultValues: {
            id: Number(user.id),
            email: user.email,
            phone: `${user.phone}`,
            password: '',
            image: null,
        },
    })

    const reloadData = async () => {
        setIsLoading(true)
        const resUser = await getCurrentUser()
        setVillaImage(resUser.profile)
        setDataUser(resUser)
        setIsLoading(false)
    }

    useEffect(() => {
        reloadData()
    },[])

    const onSubmit = async (values: z.infer<typeof formSchema>) => {
        setIsLoading(true)
        onChange(true);

        const formData = new FormData()
        formData.append('id', values.id as any);
        formData.append('email', values.email as any);
        formData.append('phone', values.phone as any);
        formData.append('password', values.password as any);
        formData.append('image', values.image as any);

        startTransition(async() => {
            try {
                const update = await updateUser(formData);
            
                if (update.success) {
                    
                    const out = await logout()

                    if(out?.success){
                        toast.success('Profile updated');
                        router.push("/login");
                    } else {
                        toast.error('Profile updated');
                        setIsLoading(false)
                    }
                } else {
                    toast.error('Failed to update profile.');
                    setIsLoading(false)
                }
            } catch (error) {
                console.error('Error updating profile:', error);
                toast.error('Something went wrong, Failed update the profile!');
                setIsLoading(false)
            }
        })
    };
    return (
        <Card className="drop-shadow-lg">
            {isLoading && ( <Loader/> )}
            <CardHeader>
                <div id="profile-1" className="flex items-center w-fit">
                    <CardTitle>{t('userProfile')}</CardTitle>
                    <div className="ml-2">
                        <ProfilePopover />
                    </div>
                </div>
                <CardDescription>{t('profileDesc')}</CardDescription>
            </CardHeader>
            <CardContent>
                <div className="grid lg:grid-cols-6 grid-cols-1 gap-8 lg:divide-x">
                    <aside id="profile-2" className="lg:col-span-2 h-full">
                        <div className="sticky top-10">
                            <Form {...form}>
                                <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
                                    <div className="flex items-center justify-end">
                                        <Button id="profile-3" type="button" variant='ghost' size='sm' onClick={() => setEdit((prev) => !prev)}>
                                            <RiEditFill className="mr-2"/>
                                            <span>Edit</span>
                                        </Button>
                                    </div>
                                    <Avatar className="h-32 w-32 mx-auto group">
                                        <AvatarImage src={villaImage} />
                                        <AvatarFallback>CN</AvatarFallback>
                                        {edit && (
                                            <div className="group-hover:block group-hover:left-0 group-hover:opacity-100 duration-500 absolute top-0 left-1/2 z-[999] w-full h-full transition-all ease-in-out opacity-0">
                                                <div className="bg-black/50 flex items-center justify-center h-full w-full">
                                                    <FormField
                                                        control={form.control}
                                                        name="image"
                                                        render={({ field: { value, onChange, ...fieldProps } }) => (
                                                            <FormItem>
                                                                <FormLabel><IoCamera className="h-10 w-10 text-white"/></FormLabel>
                                                                <FormControl>
                                                                    <Input 
                                                                        {...fieldProps} 
                                                                        type="file"
                                                                        className="hidden" 
                                                                        accept="image/*, application/pdf"
                                                                        onChange={(event) => {
                                                                            const displayUrl = URL.createObjectURL(event.target.files![0])
                                                                            const selectedFile = event.target.files![0];
                                                                            if (selectedFile) {
                                                                                form.setValue('image', selectedFile);
                                                                            } else {
                                                                                form.setValue('image', null);
                                                                            }
                                                                            setVillaImage(displayUrl);
                                                                        }}/>
                                                                </FormControl>
                                                            </FormItem>
                                                        )}
                                                    />
                                                </div>
                                            </div>
                                        )}
                                    </Avatar>
                                    <FormField
                                        control={form.control}
                                        name="id"
                                        render={({ field }) => (
                                            <FormItem>
                                                <FormControl>
                                                    <Input 
                                                        {...field} 
                                                        // value={user.id} 
                                                        className="hidden" 
                                                        disabled={!edit}/>
                                                </FormControl>
                                            </FormItem>
                                        )}
                                    />
                                    <div className="space-y-2">
                                        <FormField
                                            control={form.control}
                                            name="email"
                                            render={({ field }) => (
                                                <FormItem>
                                                    <FormLabel>{t('email')}</FormLabel>
                                                    <FormControl>
                                                        <Input 
                                                            {...field}
                                                            disabled={!edit}/>
                                                    </FormControl>
                                                </FormItem>
                                            )}
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <FormField
                                            control={form.control}
                                            name="phone"
                                            render={({ field }) => (
                                                <FormItem>
                                                    {loading ? (
                                                        <>
                                                            <Skeleton className="h-5 w-[123px] rounded-full" />
                                                            <Skeleton className="h-10 min-w-[380px] rounded-full" />
                                                            <Skeleton className="h-4 w-[120px] rounded-full" />
                                                        </>
                                                    ) : (
                                                        <>
                                                            <FormLabel>{t('phone')}</FormLabel>
                                                            <FormControl>
                                                                <Input 
                                                                    {...field} 
                                                                    // value={user.phone} 
                                                                    disabled={!edit}/>
                                                            </FormControl>
                                                            <FormDescription className="text-xs text-muted-foreground/75">
                                                                ex: +62123456789
                                                            </FormDescription>
                                                        </>
                                                    )}
                                                    <FormMessage />
                                                </FormItem>
                                            )}
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <FormField
                                            control={form.control}
                                            name="password"
                                            render={({ field }) => (
                                                <FormItem>
                                                    {loading ? (
                                                        <>
                                                            <Skeleton className="h-5 w-[93px] rounded-full" />
                                                            <Skeleton className="h-10 min-w-[380px] rounded-full" />
                                                            <Skeleton className="h-4 w-[200px] rounded-full" />
                                                        </>
                                                    ) : (
                                                        <>
                                                            <FormLabel>{t('password')}</FormLabel>
                                                            <FormControl>
                                                                <Input 
                                                                    {...field} 
                                                                    placeholder="********" 
                                                                    autoCapitalize="none"
                                                                    autoComplete="password"
                                                                    autoCorrect="off"
                                                                    disabled={!edit}/>
                                                            </FormControl>
                                                            <FormDescription className="text-xs text-muted-foreground/75">
                                                                {t('suggest')}
                                                            </FormDescription>
                                                        </>
                                                    )}
                                                    <FormMessage />
                                                </FormItem>
                                            )}
                                        />
                                    </div>
                                    <div className={`${edit ? 'flex' : 'hidden'} items-center justify-end`}>
                                        <Button type="submit">
                                            {t('edit')}
                                        </Button>
                                    </div>
                                </form>
                            </Form>
                        </div>
                    </aside>
                    <div className="lg:col-span-4 space-y-4 lg:pl-6">
                        <div id="profile-4" className="space-y-2">
                            <p className="font-medium text-lg">{t('listVilla')}</p>
                            <ListVilla data={villa}/>
                        </div>
                        <div id="profile-5" className="space-y-2">
                            <div className="flex lg:flex-row flex-col lg:items-center items-start justify-between lg:space-y-0 space-y-2">
                                <p className="font-medium lg:text-lg text-base">{`${t('emailNotif')} ( ${selected.villa_name} )`}</p>
                                <AddEmail villa={selected} loading={setIsLoading} reload={reloadData} isEdit={false}/>
                            </div>
                            <Table>
                                <TableHeader className="drop-shadow-lg bg-muted-foreground/10">
                                    <TableRow>
                                        <TableHead className="w-28">ID</TableHead>
                                        <TableHead className="w-auto">Email</TableHead>
                                        <TableHead className="w-32">Action</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {dataUser.mail && dataUser.mail.length > 0 ? (
                                        dataUser.mail.map((m: any, index: number) => (
                                            <TableRow key={m.id}>
                                                <TableCell className="w-28">{index + 1}</TableCell>
                                                <TableCell className="w-auto">{m.mail}</TableCell>
                                                <TableCell className="w-32">
                                                    <Popover>
                                                        <PopoverTrigger><CgMenuRight /></PopoverTrigger>
                                                        <PopoverContent className="w-40 space-y-3">
                                                            <PopoverClose asChild>
                                                                <AddEmail villa={selected} loading={setIsLoading} reload={reloadData} isEdit={true} data={m}/>
                                                            </PopoverClose>
                                                            <PopoverClose asChild>
                                                                <DeleteEmail mail={m} loading={setIsLoading} reload={reloadData} />
                                                            </PopoverClose>
                                                        </PopoverContent>
                                                    </Popover>
                                                </TableCell>
                                            </TableRow>
                                        ))
                                    ) : (
                                        <TableRow>
                                            <TableCell className="text-center font-semibold text-muted-foreground/50" colSpan={3}>No Email notification</TableCell>
                                        </TableRow>
                                    )}
                                </TableBody>
                            </Table>
                        </div>
                        <div id="profile-7" className="space-y-2">
                            <div className="flex lg:flex-row flex-col lg:items-center items-start justify-between lg:space-y-0 space-y-2">
                                <p className="font-medium lg:text-lg text-base">{`${t('waNotif')} ( ${selected.villa_name} )`}</p>
                                <AddWhatsapp villa={selected} loading={setIsLoading} reload={reloadData} isEdit={false}/>
                            </div>
                            <Table>
                                <TableHeader className="drop-shadow-lg bg-muted-foreground/10">
                                    <TableRow>
                                        <TableHead className="w-28">ID</TableHead>
                                        <TableHead className="w-auto">Phone</TableHead>
                                        <TableHead className="w-32">Action</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {dataUser.whatsapp && dataUser.whatsapp.length > 0 ? (
                                        dataUser.whatsapp.map((w: any, index: number) => (
                                            <TableRow key={w.id}>
                                            <TableCell className="w-28">{index + 1}</TableCell>
                                            <TableCell className="w-auto">{w.phone}</TableCell>
                                            <TableCell className="w-32">
                                                <Popover>
                                                    <PopoverTrigger><CgMenuRight /></PopoverTrigger>
                                                    <PopoverContent className="w-40 space-y-3">
                                                        <PopoverClose asChild>
                                                            <AddWhatsapp villa={selected} loading={setIsLoading} reload={reloadData} isEdit={true} data={w}/>
                                                        </PopoverClose>
                                                        <PopoverClose asChild>
                                                            <DeleteWhatsapp wa={w} loading={setIsLoading} reload={reloadData}/>
                                                        </PopoverClose>
                                                    </PopoverContent>
                                                </Popover>
                                            </TableCell>
                                            </TableRow>
                                        )) 
                                    ) : (
                                        <TableRow>
                                            <TableCell className="text-center font-semibold text-muted-foreground/50" colSpan={3}>No WhatsApp notification</TableCell>
                                        </TableRow>
                                    )}
                                </TableBody>
                            </Table>
                        </div>
                    </div>
                </div>
            </CardContent>
        </Card>
    )
}

export default Profile